#!/usr/bin/env python3 
# coding: utf-8

# 
# Import Libraries
#

import nest
import numpy

import argparse

nest.set_verbosity(100)

parser = argparse.ArgumentParser(description='STDP test with Tsodyks Synapse')
#parser.add_argument('--A2', metavar='float', type=float, help='Magnitude of the second current injectin.', default=500.0)

arg = parser.parse_args()

# Best Parameter

A_in = 5.5
Tau_rec_in = 35.0
Tau_fac_in = 1000.0

# Width of random distribution of U

gamma_param = [
        [1.40, 26.5],
        [3.12, 10.1],
        [2.94, 12.6]
        ]

# Setting up the parameters
# Copy from nest-simulator homepage

h = 0.1                # simulation step size (ms)
Tau = 15.              # membrane time constant
Theta = -50.           # threshold
E_L = -65.             # reset potential of membrane potential
R = 1                  # 100 M Ohm
C = Tau / R            # Tau (ms)/R in NEST units
TauR = 2.              # refractory time
                       
Tau_psc = 2.           # time constant of PSC (= Tau_inact)
Tau_rec = Tau_rec_in   # recovery time
Tau_fac = Tau_fac_in   # facilitation time
A = A_in               # PSC weight in pA

neuron_param = {
#               "tau_m": Tau,
                "t_ref": TauR,
                "tau_syn_ex": Tau_psc,
                "tau_syn_in": Tau_psc,
                "C_m": C,
                "V_reset": E_L,
                "E_L": E_L,
                "V_m": E_L,
                "V_th": Theta
                }

syn_param = {
             "tau_psc": Tau_psc,
             "tau_rec": Tau_rec,
             "tau_fac": Tau_fac,
             "delay": 0.1,
             "weight": A,
             "u": 0.0,
             "x": 1.0
             , "Wmax": A * 2.0
             , "mu_plus": 1.0
             , "mu_minus": 1.0
             , "lambda": 0.1
             }

sigma_i=1

for idx in range(10000):

    rate = 100.0
    tmp = numpy.random.exponential(1000.0 / rate, size=2000)
    spike_train_1 = [1.0+numpy.floor(numpy.sum(tmp[0:i+1])*10)*0.1 for i in range(len(tmp))]
    tmp = numpy.random.exponential(1000.0 / rate, size=2000)
    spike_train_2 = [1.0+numpy.floor(numpy.sum(tmp[0:i+1])*10)*0.1 for i in range(len(tmp))]

    for alpha,beta in gamma_param:
     
        # Reset the simulator

        nest.ResetKernel()
        nest.SetKernelStatus({"resolution": h})
        nest.SetKernelStatus({'rng_seeds': [idx]})

        # Create objects in the system

        # poisson_gen = nest.Create("poisson_generator", 2 ,{"rate": 100.00})
        spike_gen_1 = nest.Create("spike_generator", 1 ,{'spike_times': spike_train_1})
        spike_gen_2 = nest.Create("spike_generator", 1 ,{'spike_times': spike_train_2})
        neurons = nest.Create("iaf_cond_exp", 2)
        volts0  = nest.Create('multimeter')
        volts1  = nest.Create('multimeter')

        # Initial the object

        nest.SetStatus(neurons, neuron_param)
        nest.SetStatus(volts0, {"label": "voltmeter", "withtime": True, "withgid": True, 
                                "interval": 0.1,'record_from': ['V_m', 'g_ex']})
        nest.SetStatus(volts1, {"label": "voltmeter", "withtime": True, "withgid": True, 
                                "interval": 0.1,'record_from': ['V_m', 'g_ex']})

        # Connect two Poisson generators to neurons

        # nest.Connect([poisson_gen[0]], [neurons[0]])
        # nest.Connect([poisson_gen[1]], [neurons[1]])
        nest.Connect([spike_gen_1[0]], [neurons[0]])
        nest.Connect([spike_gen_2[0]], [neurons[1]])

        # Change the weight of these two couplings

        nest.SetStatus(nest.GetConnections([spike_gen_1[0]], [neurons[0]], "static_synapse"), "weight", 5.0)
        nest.SetStatus(nest.GetConnections([spike_gen_2[0]], [neurons[1]], "static_synapse"), "weight", 5.0)

        # Connect two neurons with a STDP-Tsodyks synapse
            
        for idx in range(1):
            while True:
                U_local = numpy.random.gamma(alpha, beta) / 140.0
                if U_local < 1.0:
                    break
            syn_param_tmp = syn_param.copy()
            syn_param_tmp["U"] = U_local
            syn_param_tmp["weight"] = syn_param["weight"] 
            syn_param_tmp["Wmax"] = syn_param["Wmax"]
            syn_param_tmp.update({"model": "stdp_tsodyks_synapse"})
            nest.Connect([neurons[0]], [neurons[1]], syn_spec=syn_param_tmp)

#        nest.CopyModel("stdp_goda_tsodyks_synapse", "stdp_dyn_syn", syn_param)
#        nest.Connect([neurons[0]], [neurons[1]], syn_spec="stdp_dyn_syn")

        # Connect voltmeters to neurons

        nest.Connect(volts0, [neurons[0]])
        nest.Connect(volts1, [neurons[1]])

        # Run the simulation

        dt = 1.0
        t_start = 0.0
        t_end = 1000.0

        #for t in numpy.arange(t_start,t_end+0.5*dt,dt):
        #    nest.Simulate(dt)
        nest.Simulate(1000.0)

        tmp_stdp_syn = numpy.sum(nest.GetStatus(nest.GetConnections([neurons[0]], [neurons[1]]), "weight"))
        print(0, tmp_stdp_syn, end="\t")
    print()
    
